package jcircus.environment;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;

import jcircus.exceptions.InvalidSubTypeException;
import jcircus.exceptions.NoNameTypeAnnotationException;
import jcircus.exceptions.NotYetImplementedException;
import jcircus.util.CircusType;
import jcircus.util.Constants;
import jcircus.util.NameType;
import jcircus.util.Util;
import net.sourceforge.czt.base.ast.ListTerm;
import net.sourceforge.czt.z.ast.ConstDecl;
import net.sourceforge.czt.z.ast.Decl;
import net.sourceforge.czt.z.ast.DeclName;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.Freetype;
import net.sourceforge.czt.z.ast.NameTypePair;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.Signature;
import net.sourceforge.czt.z.ast.VarDecl;
import net.sourceforge.czt.z.util.Factory;

/**
 * Environment.java
 *
 * @author Angela Freitas
 */
public class Environment {

    /**
     * ChannelEnv
     */
    private ChannelEnv channelEnv_;
    /**
     * ChannelSetEnv
     */
    private ChannelSetEnv channelSetEnv_;
    /**
     * Current scope
     */
    private NameTypeEnv currentScope_;
    /**
     * Scope stack
     */
    private ScopeStack scopeStack_;
    /**
     * List of types in the specification
     */
    private TypeList typeList_;
    /**
     * Environment for free type.
     */
    private FreeTypeEnv freeTypesEnv_; 
    /**
     * Factory.
     */
    private Factory factory_;
    
    /**
     * Constructor
     *
     */
    public Environment() {
        this.channelEnv_ = new ChannelEnv();
        this.channelSetEnv_ = new ChannelSetEnv();
        this.currentScope_ = new NameTypeEnv();
        this.scopeStack_ = new ScopeStack();
        this.typeList_ = new TypeList();
        this.freeTypesEnv_ = new FreeTypeEnv();
        this.factory_ = new Factory();
    }
    
    /* ******************************************************************************************************
     * Methods related to currentScope, ScopeStack and TypeList
     * ******************************************************************************************************
     */
    
    /**
     * Open scope.
     *
     */
    public void openScope() {
        this.scopeStack_.push(this.currentScope_);
        this.currentScope_ = new NameTypeEnv();
    }
    
    /**
     * Close scope.
     *
     */
    public void closeScope() {
        this.currentScope_ = this.scopeStack_.pop();
    }
    /**
     * Declare the given name and type in the current scope, without adding to the type list.
     *
     * @return  CircusType  The CircusType that was inserted, with id and javaName set.
     */  
    public CircusType declareName(DeclName name, NameType nameType, CircusType circusType) {
        return this.declareName(name, nameType, circusType, false);
    }
    
    /**
     * Declare the given name and type in the current scope.
     *
     * @param   boolean     addToTypeList   If true, adds the CircusType to the type list.
     *
     * @return  CircusType  The CircusType that was inserted, with id and javaName set.
     */
    
    public CircusType declareName(DeclName name, NameType nameType, CircusType circusType, boolean addToTypeList) {
        
        CircusType existentCircusType;

        if (circusType.equals(CircusType.createCircusInteger())) {
            circusType.setJavaCircusTypeName(Constants.CLS_CIRCINT);
            
        } else if (circusType.getExpression() instanceof RefExpr) {
            String javaName = ((RefExpr) circusType.getExpression()).getRefName().toString();
            circusType.setJavaCircusTypeName(javaName);
        } else {
            //circusType.getExpression().getClass();
            //throw new RuntimeException("Type expression not allowed.");
        }
        if (addToTypeList) {
            this.typeList_.addType(circusType);
        }
        
        // Add the annotation
        Util.addNameTypeAnn(name, nameType);
        
        // Define the variable in the current scope
        this.currentScope_.put(name, circusType);
        
        return circusType;
    }
        
    /**
     * Returns the circus type of the name
     *
     * @param variable
     * @return
     */
    public CircusType getCircusType(String name) {
        
        // Searches in the current scope
        CircusType circusType = this.currentScope_.getCircusType(name);
        
        // Searches in the inner scopes
        if (circusType == null) {
            circusType = this.scopeStack_.getType(name);
        }
        
        return circusType;
    }
    
    /**
     * Returns the name type of the name.
     *
     * @param variable
     * @return
     */
    public NameType getNameType(String name) {
        
        // Searches in the current scope
        NameType nameType = this.currentScope_.getNameType(name);
        
        // Searches in the inner scopes
        if (nameType == null) {
            nameType = this.scopeStack_.getNameType(name);
        }
        
        // Name not declared
        if (nameType == null) {
            throw new NoNameTypeAnnotationException(name);
        }
        
        return nameType;
    }
    
    /**
     * Returns the type list.
     *
     * @return
     */
    public TypeList getTypeList() {
        return this.typeList_;
    }
    
    /**
     * Return true if the name is an abbreviation or axiomatic definition
     * defined in the global scope.
     *
     */
    public boolean isGlobalConstant(DeclName name) {
        
        NameTypeEnv globalScope;
        
        if (this.scopeStack_.isEmpty()) {
            // Global scope is the current scope
            globalScope = this.currentScope_;
            
        } else {
            // Global scope is the first in the scope stack
            globalScope = this.scopeStack_.elementAt(0);
        }
        
        return globalScope.containsKey(name.toString());
    }
    
    /**
     * Returns a local environment, based on the current scope and scope stack.
     *
     * Local environment here has a different meaning from that of Marcel's
     * strategy. Here, local environment contains state components, process 
     * parameters, action parameters and local variables. In Marcel's strategy,
     * local environment contains only action parameters and local variables,
     * (both are referred to as local variables.)
     *
     */
    public NameTypeEnv localEnvironment() {
        
        NameTypeEnv r = new NameTypeEnv();
        NameTypeEnv temp;
        Enumeration keys;
        DeclName name;
        NameType nameType;
        CircusType circusType;
        
        // Inserts all the declarations of the scope stack into the local environment,
        // starting with the outermost.
        // (redeclarations in an inner scope will substitute previous declarations)
        for (int i = 0; i < this.scopeStack_.size(); i++) {
            
            temp = this.scopeStack_.elementAt(i);
            keys = temp.keys();
            
            /* debug */
            //temp.print();
            
            while(keys.hasMoreElements()) {
                
                name = (DeclName) keys.nextElement();
                nameType = (NameType) name.getAnn(NameType.class);
                
                /**
                 * Types of variables that are part of the local environment:
                 * state components, process parameters, action parameters and
                 * local variables.
                 */
                if (nameType.equals(NameType.StateComponent) ||
                        nameType.equals(NameType.ProcessParam) ||
                        nameType.equals(NameType.ActionParam) ||
                        nameType.equals(NameType.LocalVariable)) {
                    
                    circusType = temp.getCircusType(name.toString());
                    
                    r.put(name, circusType);
                }
            }
        }
        
        /* debug */
        //this.currentScope_.print();
        
        // Inserts all the declarations of the current scope into the local environment
        keys = this.currentScope_.keys();
        
        while(keys.hasMoreElements()) {
            name = (DeclName) keys.nextElement();
            circusType = this.currentScope_.getCircusType(name.toString());
            r.put(name, circusType);
        }
        
        return r;
    }
    
    public Signature getCurrentScopeAsSig() {
        
        List nameTypeList = new ArrayList();

        Enumeration keys = currentScope_.keys();
        while(keys.hasMoreElements()) {
            DeclName declName = (DeclName) keys.nextElement();
            CircusType circusType = this.getCircusType(declName.toString());
            
            NameTypePair nameTypePair = factory_.createNameTypePair(declName, circusType);
            nameTypeList.add(nameTypePair);
        }
        Signature signature = this.factory_.createSignature(nameTypeList);
        
        return signature;
    }
    /* ******************************************************************************************************
     * Methods related to ChannelEnvironment
     * ******************************************************************************************************
     */
    
    public void declareProcess(String procName, ProcChanEnv procChanEnv) {
        
        if (procChanEnv == null)
            throw new NullPointerException();
        
        this.channelEnv_.put(procName, procChanEnv);
    }
    
    public ProcChanEnv getProcessChannelEnvironment(String procName) {
        return this.channelEnv_.get(procName);
    }
    
    /* ******************************************************************************************************
     * Methods related to FreeTypesEnvironment
     * ******************************************************************************************************
     */
    
    public void freeTypesEnvironmentAdd(Freetype freetype) {
        this.freeTypesEnv_.add(freetype);
    }
    
    public int freeTypesEnvironmentSize() {
        return this.freeTypesEnv_.size();
    }
    
    public Freetype freeTypesEnvironmentGet(int i) {
        return this.freeTypesEnv_.get(i);
    }
    
    public boolean isFreeType(String name) {
        return this.freeTypesEnv_.isFreeType(name);
    }
    
    public boolean isElementFreeType(DeclName name) {
        return this.freeTypesEnv_.isElementFreeType(name);
    }

    public boolean isElementFreeType(String freeType, String element) {
        return this.freeTypesEnv_.isElementFreeType(freeType, element);
    }
    
    public String getNameFreeType(DeclName name) {
        return this.freeTypesEnv_.getNameFreeType(name);
    }

    /* ******************************************************************************************************
     * Methods related to Channel Set Environment.
     * ******************************************************************************************************
     */

    public void declareChannelSet(String chanName, HashSet chanSet) {
        this.channelSetEnv_.insert(chanName, chanSet);
    }
    
    public HashSet getChannelSet(String chanName) {
        return this.channelSetEnv_.get(chanName);
    }
    
    /* ******************************************************************************************************
     * For debug
     * ******************************************************************************************************
     */
    
    public void printCurrentScope() {
        System.out.println("Current scope:");
        this.currentScope_.print();
    }
    
    public void printScopeStack() {
        System.out.println("Scope stack:");
        this.currentScope_.print();
    }
    
}